import { Request, Response, NextFunction } from 'express';
import { sportsService } from '../services';
import { sportsQuerySchema, sportIdParamSchema } from '../validation/schemas';
import { ApiError } from '../middleware/error.middleware';

export class SportsController {
  async getSports(req: Request, res: Response, next: NextFunction): Promise<void> {
    try {
      const query = sportsQuerySchema.parse(req.query);
      const sports = await sportsService.getAllSports(query.isActive);
      res.json({ sports });
    } catch (error) {
      next(error);
    }
  }

  async getSportById(req: Request, res: Response, next: NextFunction): Promise<void> {
    try {
      const { sportId } = sportIdParamSchema.parse(req.params);
      const sport = await sportsService.getSportById(sportId);

      if (!sport) {
        throw new ApiError(404, 'Sport not found', 'SPORT_NOT_FOUND');
      }

      res.json({ sport });
    } catch (error) {
      next(error);
    }
  }
}

export const sportsController = new SportsController();
