// Base entity with common fields
export interface BaseEntity {
  id: number;
  isActive: boolean;
  createdAt?: Date;
  updatedAt?: Date;
}

// Sport entity
export interface Sport extends BaseEntity {
  sportName: string;
  icon: string | null;
}

// League entity
export interface League extends BaseEntity {
  sportId: number;
  sportName: string;
  leagueName: string;
  leagueLogo: string | null;
}

// Participant types
export type ParticipantType = 'TEAM' | 'DRIVER' | 'PLAYER';

// Participant entity
export interface Participant extends BaseEntity {
  participantType: ParticipantType;
  sportId: number;
  sportName: string;
  leagueIds: number[];
  leagueNames: string[];
  teamName: string | null;
  fullName: string | null;
  shortName: string | null;
  abbreviation: string | null;
  teamLogo: string | null;
}

// WidgetType entity
export interface WidgetType extends BaseEntity {
  sportId: number;
  sportName: string;
  leagueId: number | null;
  leagueName: string | null;
  widgetName: string;
}

// Template entity
export interface Template extends BaseEntity {
  sportId: number;
  sportName: string;
  leagueId: number | null;
  leagueName: string | null;
  participantId: number | null;
  teamName: string | null;
  widgetTypeId: number;
  widgetName: string;
  templateName: string;
  darkThemeUrl: string | null;
  lightThemeUrl: string | null;
  thumbnailDark: string | null;
  thumbnailLight: string | null;
  featured: boolean;
}

// Template with embed codes (API response)
export interface TemplateWithEmbed extends Template {
  embedCodeLight: string | null;
  embedCodeDark: string | null;
}

// ApiKey entity
export interface ApiKey extends BaseEntity {
  name: string;
  keyHash: string;
  keyPrefix: string;
  expiresAt: Date | null;
  lastUsedAt: Date | null;
}

// Seed data structure
export interface SeedData {
  version: string;
  sports: (Omit<Sport, 'createdAt' | 'updatedAt' | 'icon'> & { iconUrl: string | null })[];
  leagues: (Omit<League, 'createdAt' | 'updatedAt' | 'leagueLogo'> & { iconUrl: string | null })[];
  participants: (Omit<Participant, 'createdAt' | 'updatedAt' | 'teamLogo'> & { teamLogoUrl: string | null })[];
  widgetTypes: Omit<WidgetType, 'createdAt' | 'updatedAt'>[];
  templates: (Omit<Template, 'createdAt' | 'updatedAt' | 'thumbnailDark' | 'thumbnailLight'> & { thumbnailDarkUrl: string | null; thumbnailLightUrl: string | null })[];
}

// API response types
export interface ApiResponse<T> {
  data: T;
  meta?: {
    total?: number;
    page?: number;
    limit?: number;
  };
}

export interface ApiErrorResponse {
  error: {
    code: string;
    message: string;
  };
}
