import { TemplatesService } from './templates.service';

// Force JSON store mode for tests
process.env.USE_JSON_STORE = 'true';

describe('TemplatesService', () => {
  let service: TemplatesService;

  beforeAll(() => {
    service = new TemplatesService();
  });

  describe('getTemplates', () => {
    it('should return all templates without filters', async () => {
      const templates = await service.getTemplates({});

      expect(Array.isArray(templates)).toBe(true);
      expect(templates.length).toBeGreaterThan(0);
    });

    it('should return templates sorted by name', async () => {
      const templates = await service.getTemplates({});

      const names = templates.map(t => t.templateName);
      const sorted = [...names].sort();
      expect(names).toEqual(sorted);
    });

    it('should filter by sportId', async () => {
      const allTemplates = await service.getTemplates({});
      if (allTemplates.length > 0) {
        const sportId = allTemplates[0].sportId;
        const filtered = await service.getTemplates({ sportId });

        expect(filtered.every(t => t.sportId === sportId)).toBe(true);
      }
    });

    it('should filter by leagueId', async () => {
      const allTemplates = await service.getTemplates({});
      const withLeague = allTemplates.filter(t => t.leagueId !== null);

      if (withLeague.length > 0) {
        const leagueId = withLeague[0].leagueId!;
        const filtered = await service.getTemplates({ leagueId });

        expect(filtered.every(t => t.leagueId === leagueId)).toBe(true);
      }
    });

    it('should filter by widgetTypeId', async () => {
      const allTemplates = await service.getTemplates({});
      if (allTemplates.length > 0) {
        const widgetTypeId = allTemplates[0].widgetTypeId;
        const filtered = await service.getTemplates({ widgetTypeId });

        expect(filtered.every(t => t.widgetTypeId === widgetTypeId)).toBe(true);
      }
    });

    it('should filter by featured', async () => {
      const featured = await service.getTemplates({ featured: true });

      expect(featured.every(t => t.featured === true)).toBe(true);
    });

    it('should filter by isActive', async () => {
      const active = await service.getTemplates({ isActive: true });

      expect(active.every(t => t.isActive === true)).toBe(true);
    });

    it('should include embed codes in response', async () => {
      const templates = await service.getTemplates({});

      if (templates.length > 0) {
        const template = templates[0];
        expect(template).toHaveProperty('embedCodeLight');
        expect(template).toHaveProperty('embedCodeDark');
        expect(template.embedCodeLight).toContain('<iframe');
        expect(template.embedCodeDark).toContain('<iframe');
      }
    });

    it('should generate correct embed code format', async () => {
      const templates = await service.getTemplates({});

      if (templates.length > 0) {
        const template = templates[0];

        // Check light theme embed
        expect(template.embedCodeLight).toContain(template.lightThemeUrl);
        expect(template.embedCodeLight).toContain('width="100%"');
        expect(template.embedCodeLight).toContain('height="600"');
        expect(template.embedCodeLight).toContain('frameborder="0"');

        // Check dark theme embed
        expect(template.embedCodeDark).toContain(template.darkThemeUrl);
      }
    });

    it('should return templates with correct structure', async () => {
      const templates = await service.getTemplates({});

      if (templates.length > 0) {
        const template = templates[0];
        expect(template).toHaveProperty('id');
        expect(template).toHaveProperty('sportId');
        expect(template).toHaveProperty('sportName');
        expect(template).toHaveProperty('widgetTypeId');
        expect(template).toHaveProperty('widgetName');
        expect(template).toHaveProperty('templateName');
        expect(template).toHaveProperty('darkThemeUrl');
        expect(template).toHaveProperty('lightThemeUrl');
        expect(template).toHaveProperty('featured');
        expect(template).toHaveProperty('isActive');
        expect(template).toHaveProperty('embedCodeLight');
        expect(template).toHaveProperty('embedCodeDark');
      }
    });

    it('should support multiple filters', async () => {
      const allTemplates = await service.getTemplates({});
      if (allTemplates.length > 0) {
        const sportId = allTemplates[0].sportId;
        const filtered = await service.getTemplates({ sportId, isActive: true });

        expect(filtered.every(t => t.sportId === sportId && t.isActive === true)).toBe(true);
      }
    });
  });
});
